/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.log;

import javax.ide.Service;
import javax.ide.command.Context;
import javax.ide.spi.ProviderNotFoundException;

/**
 * The LogManager provides the interface through which the IDE manipulates 
 * {@link LogPage}s.<p>
 */
public abstract class LogManager extends Service
{
  /**
   * Pseudo-class identifying the message page.
   */
  public static final String MSG_PAGE_CLASS = "__jsr198MessagePage";

  /**
   * Open and display the page identified by the given type identifier.
   *
   * @param context The current {@link Context}.
   * @param pageClass The page class.
   *
   * @return The log page identified by the specified page identifier.
   */
  public abstract LogPage openPage( Context context, String pageClass );
  
  /**
   * Find the log page with the specified id. 
   * 
   * @param pageClass the log page class.
   * @return the log page instance if the page is already open, or null if
   *    no log page with the specified id is open.
   */
  public abstract LogPage findPage( String pageClass );

  /**
   * Close the specified <CODE>page</CODE>.
   *
   * @param page the page to closed.
   */
  public abstract void closePage( LogPage page );

  /**
   *  Get the message page. The message page can be used as the 
   *  default page to display textual messages for the user.
   *
   *  @return get the default message page.
   */
  public abstract LogPage getMsgPage();

  /** 
   * Gets the currently selected page.
   *
   * @return the currently selected page.
   */
  public abstract LogPage getSelectedPage();
  
  protected final void initialize()
  {
    
  }
  
  /**
   * Get the log manager implementation for this IDE.
   * 
   * @return return a LogManager implementation.
   */
  public static LogManager getLogManager()
  {
    try
    {
      return (LogManager) getService( LogManager.class );
    }
    catch ( ProviderNotFoundException nse )
    {
      nse.printStackTrace();
      throw new IllegalStateException( "No log manager" );
    }
  }
}
