/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.build.spi;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;
import javax.ide.extension.MetaClassVisitor;
import javax.ide.util.MetaClass;

/**
 * Build system information gathered from processing the 
 * <b>build-system-hook</b> section of extension manifests. 
 */
public final class BuildSystemHook extends ExtensionHook
{
  private static final ElementName LISTENER_CLASS = 
    new ElementName( MANIFEST_XMLNS, "listener-class" );

  private static final ElementName BUILD_LISTENERS =
    new ElementName( MANIFEST_XMLNS, "build-listeners" );

  public static final ElementName ELEMENT = 
    new ElementName( MANIFEST_XMLNS, "build-system-hook" );

  private final Collection _metaListeners = new ArrayList();

  private final ElementVisitor _listenerVisitor = new ListenerClassVisitor();
  private final ElementVisitor _buildListenersVisitor = new BuildListenersVisitor();

  /**
   * Get the list of build system listeners 
   * declared in an extension manifest. This information records 
   * the build system listener classes interested in receiving events from the 
   * build system.
   *
   * @return a collection of {@link javax.ide.build.BuildListener}s.
   */
  public Collection getListeners()
  {
    List listeners = new ArrayList( _metaListeners.size() );
    for ( Iterator i = _metaListeners.iterator(); i.hasNext(); )
    {
      MetaClass thisClass = (MetaClass) i.next();
      try
      {
        listeners.add( thisClass.newInstance() );
      }
      catch ( Exception e )
      {
        // TODO: Decide what to do with exceptions like this...
        e.printStackTrace();
      }
    }
    return listeners;
  }
  
  public void start( ElementStartContext ctx )
  {
    ctx.registerChildVisitor( BUILD_LISTENERS, _buildListenersVisitor );
  }
  
  private class BuildListenersVisitor extends ElementVisitor
  {
    public void start( ElementStartContext context )
    {
      context.registerChildVisitor( LISTENER_CLASS, _listenerVisitor );
    }
  }
  
  private class ListenerClassVisitor extends MetaClassVisitor
  {
    protected void metaClass( ElementContext context, MetaClass mc )
    {
      _metaListeners.add( mc );
    }
  }
  
  

}
