/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.wizard.spi;

import javax.ide.util.IconDescription;
import javax.ide.util.MetaClass;

/**
 * Record of information identifying a wizard type. The information used 
 * to initialize this class comes from an Extension Deployment Descriptor
 * (EDD).<p>
 *
 * The information kept in this class includes the wizard label,
 * icon, and brief description that can be displayed to the user.
 */
public final class WizardInfo
{
  private String _label;
  private IconDescription _icon;
  private String _toolTip;
  private final MetaClass _class;
  private WizardCategory _parent = WizardCategory.NONE;

  /**
   * Constructor.
   */
  WizardInfo( MetaClass wizardClass ) 
  {
    _class = wizardClass;
  }
  
  /**
   * Get the wizard class.
   * 
   * @return the wizard class.
   */
  public MetaClass getWizardClass()
  {
    return _class;
  }
  
  /**
   * Get the category of this wizard, if any.
   * 
   * @return the parent category of this wizard if any. May return 
   *    {@link WizardCategory#NONE} to indicate that it is at the root or 
   *    does not have a category.
   */
  public WizardCategory getCategory()
  {
    return _parent;
  }
  
  void setCategory( WizardCategory cat )
  {
    _parent = cat;
  }

  /**
   * Get a localizable label describing the wizard. This label is for user
   * display.
   * @return a label describing this class. 
   */
  public String getLabel()
  {
    return _label;
  }

  /**
   * Set a localizable label describing the wizard. This label is for user
   * display.
   * @param label label describing this wizard. 
   */
  void setLabel( String label )
  {
    _label = label;
  }

  /**
   * Get a localizable icon path that can be used to graphically 
   * represent what this wizard creates.
   * @return the icon path to an image for user display.
   */
  public IconDescription getIcon()
  {
    return _icon;
  }

  /**
   * Set a localizable icon path that can be used to graphically 
   * represent what this wizard creates.
   * @param iconPath the icon path to an image for user display.
   */
  void setIcon( IconDescription icon )
  {
    _icon = icon;
  }

  /**
   * Get a description of what this wizard does.  This descriptions is for 
   * user display.
   * @return a description of the wizard.
   */
  public String getToolTip()
  {
    return _toolTip;
  }

  /**
   * Set a description of what this wizard does.  This descriptions is for 
   * user display.
   * @param toolTip a description of the wizard.
   */
  void setToolTip( String toolTip )
  {
    _toolTip = toolTip;
  }
}
