/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.menu.spi;

import java.util.logging.Level;

import javax.ide.command.Controller;
import javax.ide.extension.BooleanVisitor;
import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementEndContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;
import javax.ide.extension.I18NCharVisitor;
import javax.ide.extension.I18NStringVisitor;
import javax.ide.extension.IconVisitor;
import javax.ide.extension.MetaClassVisitor;
import javax.ide.menu.IDEAction;
import javax.ide.util.IconDescription;
import javax.ide.util.MetaClass;

final class ActionsVisitor extends ElementVisitor
{
  static final String KEY_ACTION = "action"; 

  private static final ElementName ACTION = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "action" );
  private static final ElementName LABEL = new ElementName( 
    ExtensionHook.MANIFEST_XMLNS, "label" );
  private static final ElementName MNEMONIC = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "mnemonic" );
  private static final ElementName TOOLTIP = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "tooltip" );
  private static final ElementName ICONPATH = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "iconpath");
  private static final ElementName ENABLED = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "enabled" );
  private static final ElementName CHECKED = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "checked" );
  private static final ElementName INVOKE_CLASS = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "invoke-class" );
  private static final ElementName UPDATE_CLASS = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "update-class" );    
  private static final ElementName ACTION_OVERRIDE = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "action-override" );
    
  private ElementVisitor _actionHandler = new ActionVisitor();

  private ElementVisitor _labelHandler = new LabelHandler();
  private ElementVisitor _mnemonicHandler = new MnemonicHandler();
  private ElementVisitor _tooltipHandler = new TooltipHandler();
  private ElementVisitor _iconpathHandler = new IconpathHandler();
  private ElementVisitor _enabledHandler = new EnabledHandler();
  private ElementVisitor _checkedHandler = new CheckedHandler();
  private ElementVisitor _invokeClassVisitor = new InvokeClassVisitor();
  private ElementVisitor _updateClassVisitor = new UpdateClassVisitor();
  private ElementVisitor _actionOverrideVisitor = new ActionOverrideVisitor();
  
  private static final String KEY_CONTROLLER_HOLDER = "controllerHolder";
  private static final String KEY_ACTION_REF = "actionRef";

  public void start( ElementStartContext context )
  {
    context.registerChildVisitor( ACTION, _actionHandler );
    context.registerChildVisitor( ACTION_OVERRIDE, _actionOverrideVisitor );
  }
  
  
  private class ActionVisitor extends ElementVisitor
  {
    public void start( ElementStartContext start )
    {
      String id = start.getAttributeValue( "id" ).trim();
      if ( id == null )
      {
        log( start, Level.SEVERE, "Missing attribute 'id'" );
      }
      else
      {
        MenuModel model = 
          (MenuModel) start.getScopeData().get( MenuHook.KEY_MENU_MODEL );
        if ( !model.isUniqueActionId( id ) )
        {
          log( start, Level.SEVERE, "Duplicate definition of action id '"+id+"'" );
          return;
        }
      
        String toggleText = start.getAttributeValue( "toggle" );
        boolean isToggle = toggleText == null ? false : 
          Boolean.valueOf( toggleText.trim() ).booleanValue();
        
        IDEAction action = new IDEAction( id );
        action.setToggleItem( isToggle );
        
        start.getScopeData().put( KEY_ACTION, action );
        start.getScopeData().put( KEY_CONTROLLER_HOLDER, new ControllerHolder() );
        
        start.registerChildVisitor( LABEL, _labelHandler );
        start.registerChildVisitor( MNEMONIC, _mnemonicHandler );
        start.registerChildVisitor( TOOLTIP, _tooltipHandler );
        start.registerChildVisitor( ICONPATH, _iconpathHandler );
        start.registerChildVisitor( ENABLED, _enabledHandler );
        start.registerChildVisitor( CHECKED, _checkedHandler );
        start.registerChildVisitor( UPDATE_CLASS, _updateClassVisitor );
        start.registerChildVisitor( INVOKE_CLASS, _invokeClassVisitor );
      }
    
    }
    
    public void end( ElementEndContext context )
    {
      IDEAction action = getAction( context );
      ControllerHolder holder = 
        (ControllerHolder) context.getScopeData().get( KEY_CONTROLLER_HOLDER );
      
      if ( action.getLabel() == null )
      {
        log( context, Level.SEVERE, "Required element 'label' missing." );
        return;   // Return without registering the action.
      }
      
      
      MenuModel model = 
        (MenuModel) context.getScopeData().get( MenuHook.KEY_MENU_MODEL );
      
      if ( holder.invokeClass != null )
      {
        Controller controller = new Controller( holder.invokeClass, 
          holder.updateClass );
        model.addController( action.getID(), controller );
      }

      model.addAction( getAction( context ) );
    }
  }
  
  private class ActionOverrideVisitor extends ElementVisitor
  {
    public void start( ElementStartContext context )
    {
      String actionRef = context.getAttributeValue( "action-ref" );
      if ( actionRef == null || (actionRef = actionRef.trim()).length() == 0)
      {
        log( context, Level.SEVERE, 
          "Missing required attribute 'action-ref'." );
        return;
      }
      
      context.getScopeData().put( KEY_ACTION_REF, actionRef );
      context.getScopeData().put( KEY_CONTROLLER_HOLDER, new ControllerHolder() );
    
      context.registerChildVisitor( UPDATE_CLASS, _updateClassVisitor );
      context.registerChildVisitor( INVOKE_CLASS, _invokeClassVisitor );
    }
    
    public void end( ElementEndContext context )
    {
      ControllerHolder holder = 
        (ControllerHolder) context.getScopeData().get( KEY_CONTROLLER_HOLDER );
      
      if ( holder.invokeClass == null )
      {
        log( context, Level.SEVERE, 
          "Missing required element 'invoke-class'." );
        return;
      }
      
      String actionRef = (String) context.getScopeData().get( KEY_ACTION_REF );
      MenuModel model = 
        (MenuModel) context.getScopeData().get( MenuHook.KEY_MENU_MODEL );
      
      model.addController( actionRef, new Controller( 
        holder.invokeClass, holder.updateClass
      ) );      
      
      
    }
  }
  
  private IDEAction getAction( ElementContext context )
  {
    return (IDEAction) context.getScopeData().get( KEY_ACTION );
  }
  
  private class LabelHandler extends I18NStringVisitor
  {
    public void string( ElementContext context, String text )
    {
      getAction( context ).setLabel( text );
    }
  }
  
  private class MnemonicHandler extends I18NCharVisitor
  {
    public void characterValue( ElementContext context, char theChar )
    {
      getAction( context ).setMnemonic( theChar );
    }
  }
  
  private class TooltipHandler extends I18NStringVisitor
  {
    public void string( ElementContext context, String text )
    {
      getAction( context ).setToolTip( text );
    }
  }
  
  private class IconpathHandler extends IconVisitor
  {
    public void icon( ElementContext context, IconDescription icon )
    {
      getAction( context ).setIcon( icon );
    }
  }
  
  private class EnabledHandler extends BooleanVisitor
  {
    public void booleanValue( ElementContext context, boolean value )
    {
      getAction( context ).setEnabled( value );
    }
  }
  
  private class CheckedHandler extends BooleanVisitor
  {
    public void booleanValue( ElementContext context, boolean value )
    {
      getAction( context ).setChecked( value );
    }
  }
  
  private class UpdateClassVisitor extends MetaClassVisitor
  {
    public void metaClass(ElementContext context, MetaClass mc)
    {
      ControllerHolder ch = (ControllerHolder) context.getScopeData().get(
        KEY_CONTROLLER_HOLDER
      );
      ch.updateClass = mc;
    }
  }
  
  private class InvokeClassVisitor extends MetaClassVisitor
  {
    public void metaClass(ElementContext context, MetaClass mc)
    {
      ControllerHolder ch = (ControllerHolder) context.getScopeData().get(
        KEY_CONTROLLER_HOLDER
      );
      ch.invokeClass = mc;
    }
  }
  
  private final class ControllerHolder
  {
    private MetaClass invokeClass;
    private MetaClass updateClass;
  }
}
