/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.ide.build.spi.BuildSystemHook;
import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;
import javax.ide.log.spi.LogHook;
import javax.ide.menu.spi.MenuHook;
import javax.ide.model.spi.DocumentHook;
import javax.ide.property.spi.PropertyHook;
import javax.ide.spi.IDEListenerHook;
import javax.ide.wizard.spi.WizardHook;
import javax.ide.net.spi.VFSHook;
import javax.ide.editor.spi.EditorHook;
import javax.ide.extension.ElementVisitorFactory2;
import javax.ide.extension.Extension;
import javax.ide.extension.UnrecognizedElementException;

/**
 * The default implementation of the visitor factory for hooks. This
 * implementation returns the default implementations of all standard
 * hooks defined by the JSR-198 spec and provides a registerHookVisitor()
 * method used by the hook-handler-hook to register custom hooks declared
 * in the manifest.
 */
public class DefaultHookVisitorFactory implements ElementVisitorFactory2 
{
  private final static Logger LOG = Logger.getLogger( DefaultHookVisitorFactory.class.getName() );

  private final Map<ElementName,ExtensionHook> _hookVisitors = new HashMap<ElementName,ExtensionHook>();

  /**
   * Constructs the default hook visitor factory. This implementation calls
   * {@link #registerStandardVisitors()}.
   */
  public DefaultHookVisitorFactory()
  {
    registerStandardVisitors();
  }
  
  /**
   * Get all registered hooks.
   * 
   * @return all registered hooks.
   */
  public final Collection<ExtensionHook> getHooks()
  {
    return Collections.unmodifiableCollection( _hookVisitors.values() );
  }
  
  /**
   * Register hooks defined in the JSR-198 spec. Unlike user defined hooks,
   * these are always present and hence statically registered rather than
   * registered via an extension manifest. <p>
   * This implementation calls the following methods in sequence:<br />
   * <ul>
   * </ul>
   */
  protected final void registerStandardVisitors()
  {
    registerHook( HookHandlerHook.ELEMENT, createHookHandlerHook() );
    registerHook( BuildSystemHook.ELEMENT, createBuildSystemHook() );
    registerHook( FeatureHook.ELEMENT, createFeatureHook() );
    registerHook( LogHook.ELEMENT, createLogHook() );
    registerHook( MenuHook.ELEMENT, createMenuHook() );
    registerHook( DocumentHook.ELEMENT, createDocumentHook() );
    registerHook( PropertyHook.ELEMENT, createPropertyHook() );
    registerHook( IDEListenerHook.ELEMENT, createIdeListenerHook() );
    registerHook( WizardHook.ELEMENT, createWizardHook() );
    registerHook( VFSHook.ELEMENT, createVFSHook() );
    registerHook( EditorHook.ELEMENT, createEditorHook() );
  }
  
  protected ExtensionHook createEditorHook()
  {
    return new EditorHook();
  }
  
  /**
   * Create the VFS hook. This implementation returns a new instance of
   * javax.ide.net.spi.VFSHook.
   * 
   * @return the VFS hook.
   */
  protected ExtensionHook createVFSHook()
  {
    return new VFSHook();
  }
  
  /**
   * Create the hook handler hook. This implementation returns a new instance
   * of javax.ide.extension.spi.HookHandlerHook.
   * 
   * @return the hook handler hook.
   */
  protected ExtensionHook createHookHandlerHook()
  {
    return new HookHandlerHook( this );
  }
  
  /**
   * Create the wizard hook. This implementation returns a new instance of 
   * javax.ide.wizard.spi.WizardHook.
   * 
   * @return the wizard hook.
   */
  protected ExtensionHook createWizardHook()
  {
    return new WizardHook();
  }
  
  /**
   * Create the IDE Listener hook. This implementation returns a new instance of
   * javax.ide.spi.IDEListenerHook.
   * 
   * @return the ide listener hook.
   */
  protected ExtensionHook createIdeListenerHook()
  {
    return new IDEListenerHook();
  }
  
  /**
   * Create the property hook. This implementation returns a new instance of
   * javax.ide.property.spi.PropertyHook.
   * 
   * @return the property hook.
   */
  protected ExtensionHook createPropertyHook()
  {
    return new PropertyHook();
  }
  
  /**
   * Create the document hook. This implementation returns a new instance
   * of javax.ide.model.spi.DocumentHook.
   * 
   * @return the document hook.
   */
  protected ExtensionHook createDocumentHook()
  {
    return new DocumentHook();
  }
  
  /**
   * Create the menu hook. This implementation returns a new instance of
   * javax.ide.menu.spi.MenuHook.
   * 
   * @return the menu hook.
   */
  protected ExtensionHook createMenuHook()
  {
    return new MenuHook();
  }
  
  /**
   * Create the log hook. This implementation returns a new instance of
   * javax.ide.extension.spi.LogHook.
   * 
   * @return the log hook.
   */
  protected ExtensionHook createLogHook()
  {
    return new LogHook();
  }

  /**
   * Create the feature hook. This implementation returns a new instance of
   * javax.ide.extension.spi.FeatureHook.
   * 
   * @return the feature hook.
   */
  protected ExtensionHook createFeatureHook()
  {
    return new FeatureHook();
  }

  /**
   * Create the build system hook. This implementation returns a new instance
   * of javax.ide.build.spi.BuildSystemHook.
   * 
   * @return the build system hook.
   */
  protected ExtensionHook createBuildSystemHook()
  {
    return new BuildSystemHook();
  }

  protected ElementVisitor validateHookForExtension( ElementContext context,
                                                     ElementName name, 
                                                     ExtensionHook hook, 
                                                     Extension currentExtension )
      throws UnrecognizedElementException
  {
    
    if (hook == null)
    {
      throw new UnrecognizedElementException( 
        "'" + name.getLocalName() + "' in namespace '" + name.getNamespaceURI() + 
        "' is not a recognized hook."
      );
    }
    
    if (currentExtension == null)
    {
      LOG.log( Level.WARNING, "No extension in context for element " + name );
      return getVisitor( name );
    }
  
    
    validateExtensionHasDependencyOnHookProvider(context,
                                                 name,
                                                 hook,
                                                 currentExtension);
    
    return hook;
  }
  
  protected void validateExtensionHasDependencyOnHookProvider(ElementContext context,
                                                              ElementName name, 
                                                              ExtensionHook hook, 
                                                              Extension currentExtension) 
    throws UnrecognizedElementException
  {
    
    final String originatingId = hook.getProvider();
    
    if ( originatingId == null || originatingId.equals( currentExtension.getID() ) )
    {
      // OK, it's a "standard" hook with no originating extension. No 
      // dependency on the provider is required.
      return;
    }
    
    /*
    * The code that follows causes a large performance problem
    * during startup. It increases the extension manifest processing
    * time by a factor of 2. The code is commented out while we
    * come up with a better algorithm.
    *
    if ( currentExtension instanceof DefaultExtension )
    {
      for ( String id : ((DefaultExtension)currentExtension).getAllImportedExtensions( ExtensionRegistry.getExtensionRegistry() ) )
      {
        if ( originatingId.equals( id ) )
        {
          return;
        }
      }
      throw new UnrecognizedElementException(
        "Must import extension '" + originatingId + "' to use hook '" + 
        name.getLocalName() + "' in namespace '" + name.getNamespaceURI() + "'."
      );
    }

    *
    * End of badly performing code.
    */
  }
  

  public final ElementVisitor getVisitor(ElementContext context,
                                   ElementName name)
    throws UnrecognizedElementException
  {

    final ExtensionHook hook = _hookVisitors.get( name );
    final Extension currentExtension = context.getExtension();
    
    return validateHookForExtension( context, name, hook, currentExtension );
  }


  public final ElementVisitor getVisitor( ElementName name )
  {
    return _hookVisitors.get( name );
  }
  
  /**
   * Get whether there is a registered visitor for the specified hook
   * element name.
   * 
   * @param name the element name of a hook.
   * @return true if a visitor is registered for the specified element 
   *    name.
   */
  public boolean isNameRegistered( ElementName name )
  {
    return _hookVisitors.containsKey( name );
  }
  
  /**
   * Register the specified hook visitor. This is typically used by the 
   * hook-handler-hook to register custom hooks provided by extension
   * developers.
   * 
   * @param name a qualified element name.
   * @param hook the hook visitor to use for the specified element name.
   */
  public final void registerHook( ElementName name, ExtensionHook hook )
  {
    _hookVisitors.put( name, hook );
  }

  public boolean isDescending()
  {
    return false;
  }
}
