/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;

import java.io.IOException;
import java.io.InputStream;

import java.net.URI;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.ide.Service;
import javax.ide.extension.spi.DefaultElementContext;
import javax.ide.extension.spi.DefaultHookVisitorFactory;
import javax.ide.extension.spi.DependencyTree;
import javax.ide.extension.spi.ExtensionSource;
import javax.ide.extension.spi.ExtensionVisitor;
import javax.ide.extension.spi.MinimalExtensionParser;
import javax.ide.extension.spi.MinimalExtensionVisitor;
import javax.ide.extension.spi.PullManifestParser;
import javax.ide.spi.ProviderNotFoundException;
import javax.ide.util.Version;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.stream.XMLStreamException;


/**
 * The extension registry provides access to information about installed
 * extensions.
 */
public abstract class ExtensionRegistry extends Service
{
  private Map<String,Extension> _extensions;
  private ElementVisitorFactory _hookFactory;
  private ExtensionVisitor _extensionVisitor;
  private Map<String,Extension> _idToMinimalExtensionMap;
  
  /**
   * Find the {@link Extension} identified by the given <code>id</code>.<p>
   * 
   * @param id the id of an extension.
   * @return the specified extension, or null if no such extension is
   *    registered.
   */
  public final Extension findExtension( String id )
  {
    if ( _extensions == null )
    {
      // We are still processing extensions. We have to look up the extension
      // using the visitor's list of extensions.
      return _extensionVisitor != null ? _extensionVisitor.findExtension( id ) : null;
    }
    return _extensions.get( id );
  }
  

  /**
   * Get a collection of registered extensions.
   *
   * @return a collection of all registered extensions.
   */
  public final Collection<Extension> getExtensions()
  {
    return Collections.unmodifiableCollection( _extensions.values() );
  }
  
  /**
   * Get the hook for the specified element name.
   * 
   * @param hookElement the element name of a hook to retrieve. Must not be
   *    null.
   * @return the hook for the specified element, or null if no such hook
   *    is defined.
   */
  public ExtensionHook getHook( ElementName hookElement )
  {
    if ( hookElement == null )
    {
      throw new NullPointerException( "Null hookElement" );
    }
    try
    {
      return (ExtensionHook) _hookFactory.getVisitor( hookElement );
    }
    catch ( UnrecognizedElementException ue )
    {
      return null;
    }
  }

  /**
   * Get the minimal Extension object for the given extension id. 
   */
  public Extension findMinimalExtension(String extensionId)
  {
    return _idToMinimalExtensionMap.get(extensionId);
  }   
  
  
  /**
   * Find all valid extension sources. A source is typically a JAR file 
   * containing an extension manifest in its META-INF directory. However, this
   * implementation is not limited to processing extensions bundled that way.
   * 
   * @return a collection of ExtensionSource instances, one for every potential
   *    extension.
   */
  protected abstract Collection findAllExtensionSources();
  
  /**
   * Create the initial parsing context. For the SAXManifestParser, this must
   * be an instance of DefaultElementContext.
   * 
   * @return the initial parsing context.
   */
  protected ElementContext createInitialContext()
  {
    return new DefaultElementContext();
  }
  
  /**
   * Loads all extensions. <p>
   * 
   * This implementation obtains the collection of extension sources by calling
   * {@link #findAllExtensionSources()}. It then refines this list to only
   * sources which are valid (have a non-null URI).<p>
   * 
   * This implementation performs dependency analysis of extensions to determine
   * the correct load order. Given multiple extensions with the same ID but
   * different versions, it will always choose the latest version (even if that
   * causes a dependency from another extension to be unsatisfied).<p>
   * 
   * When the list of extensions to load in order is determined, this
   * implementation calls {@link #loadExtensions( Collection )}. It also calls
   * {@link #cycleEncountered( Collection )} if any cyclic dependencies were
   * found and {@link #unsatisfiedExtensionDependencies( Extension, Collection )}
   * if unsatisfied dependencies were found.
   * 
   * @return a collection of all available extensions.
   */
  protected Collection loadExtensions()
  {
    Collection sources = findAllExtensionSources();
    
    List<ExtensionSource> validSources = new ArrayList<ExtensionSource>();
    List<ExtensionSource> failedSources = new ArrayList<ExtensionSource>();  
    for ( Iterator i = sources.iterator(); i.hasNext(); )
    {
      ExtensionSource source = (ExtensionSource) i.next();
      if ( source.getManifestURI() != null )
      {
        validSources.add( source );
      }
    }
    
    MinimalExtensionVisitor visitor = createMinimalExtensionVisitor();
    MinimalExtensionParser.loadMinimal(visitor, 
                                      (DefaultElementContext)createInitialContext(),
                                      validSources,
                                      failedSources);
    
    _idToMinimalExtensionMap = visitor.getIdToMinimalExtensionMap(); 
    Map<Extension,ExtensionSource> minimalExtensionsToSources = visitor.getSourcesByExtension();
    
    Iterator<Extension> iter = minimalExtensionsToSources.keySet().iterator();
    while (iter.hasNext())
    {
      Extension minimalExtension = iter.next();
      if (!isExtensionEnabled(minimalExtension.getID(), minimalExtension.getVersion()))
      {
        //Remove the disabled extensions
        iter.remove();
      }
    }
    
    // Add extensio
    
    Sorter extensionSorter = createExtensionSorter(minimalExtensionsToSources);
    List<ExtensionSource> orderedSources = extensionSorter.getOrderedExtensionSources();
  
    Collection extensions = loadExtensions( orderedSources );
    
    extensionSorter.handleCycles();
    extensionSorter.handleUnsatisfiedDependencies();

    return extensions;
  }
  
  public static interface Sorter {
      List<ExtensionSource> getOrderedExtensionSources();
      void handleCycles();
      void handleUnsatisfiedDependencies();
  }
  
  private class DefaultSorter implements Sorter {
    private DependencyTree _dependencyTree;
    
    DefaultSorter(Map<Extension, ExtensionSource> minimalExtensionsToSources) {
      _dependencyTree = DependencyTree.buildTree( minimalExtensionsToSources );
    }

    @Override
    public List<ExtensionSource> getOrderedExtensionSources()
    {
      ArrayList<ExtensionSource> orderedSources = new ArrayList<ExtensionSource>();
      for ( Iterator i = _dependencyTree.getSortedExtensionIDs().iterator(); i.hasNext(); )
      {
        String id = (String) i.next();
        orderedSources.add( _dependencyTree.getSource( id ) );
      }
      return orderedSources;
    }

    @Override
    public void handleCycles()
    {
      for ( Iterator i = _dependencyTree.getCycles().iterator(); i.hasNext(); )
      {
        cycleEncountered( (Collection) i.next() );
      }
    }

    @Override
    public void handleUnsatisfiedDependencies()
    {
      for ( Iterator i = _dependencyTree.getUnsatisfiedExtensions().iterator(); i.hasNext(); )
      {
        Extension unsatisfied = (Extension) i.next();
        Collection deps = _dependencyTree.getUnsatisfiedDependencies( unsatisfied );
        unsatisfiedExtensionDependencies( unsatisfied, deps );
      }
    }
  }
  
  protected Sorter createExtensionSorter(Map<Extension,ExtensionSource> minimalExtensionsToSources) {
    return new DefaultSorter(minimalExtensionsToSources);
  }
  
  /**
   * Creates the MinimalExtensionVisitor instance used for the minimal
   * parse of all extension sources.  
   * 
   */
  protected MinimalExtensionVisitor createMinimalExtensionVisitor()
  {
    return new MinimalExtensionVisitor();
  }
  
  /**
   * Gets whether the specified extension is enabled in this IDE. <p>
   * 
   * This implementation always returns true.
   * 
   * @param id the id of an extension
   * @param version the version of an extension
   * @return true if the specified extension is enabled in this IDE.
   */
  protected boolean isExtensionEnabled( String id, Version version )
  {
    return true;
  }
  
  /**
   * A cycle was encountered. This is called after all extensions have been loaded to 
   * notify the extension registry that there were cyclic dependencies between
   * extensions. The extension loading implementation in this class will 
   * load extensions even if there are cycles. cycleEncountered() is called
   * to give IDEs a chance to report cycles to users.<p>
   * 
   * The returned collection contains partially populated Extension instances in
   * dependency order. The last extension in the collection is a duplicate of 
   * one of the previous extensions in the collection and is the first extension
   * on which a cycle was detected.<p>
   * 
   * This implementation does nothing.
   *
   * @param cycle a collection of Extensions that have a cyclic dependency.
   */
  protected void cycleEncountered( Collection cycle )
  {
    
  }
  
  /**
   * An extension with unsatisfied dependencies was encountered. This is called
   * after all extensions have been loaded to notify the extension registry that
   * there were extensions which had unsatisfied dependencies. The extension
   * loading implementation in this class will not load extensions which have
   * unsatisfied dependencies. unsatisfiedExtensionDependencies() is called to
   * give IDEs a chance to report unsatisfied dependencies to users.<p>
   * 
   * This implementation does nothing.
   * 
   * @param ext an extension which was not loaded due to unresolved 
   *    dependencies on other extensions.
   * @param deps a collection of <tt>ExtensionDependency</tt> objects, one for
   *    each dependency that was not satisfied.
   */
  protected void unsatisfiedExtensionDependencies( Extension ext, 
    Collection deps )
  {
     
  }
  
  /**
   * Load extensions in the specified order.<p>
   * 
   * This implementation uses SAXManifestParser to load each extension in 
   * turn.
   * 
   * @param orderedSources a collection of ExtensionSource instances. The
   *    iterator order of this collection is the correct order to load the
   *    extensions based on their dependencies.
   * @return a collection of fully loaded Extensions.
   */
  protected Collection loadExtensions( Collection orderedSources )
  {
    if ( _hookFactory == null )
    {
      _hookFactory = createHookVisitorFactory();
    }

    DefaultElementContext context = (DefaultElementContext) createInitialContext();
    PullManifestParser parser = new PullManifestParser(context);
    _extensionVisitor = createExtensionVisitor( _hookFactory );
    parser.getContext().registerChildVisitor( ExtensionVisitor.ELEMENT, 
      _extensionVisitor );
    
    int totalCount=orderedSources.size();
    int currentCount=1;
    
    for ( Iterator i = orderedSources.iterator(); i.hasNext(); )
    {
      ExtensionSource source = (ExtensionSource) i.next();
            
      loadExtension(parser, context.getLogger(), source, totalCount, currentCount);
      currentCount++;
    }
    return _extensionVisitor.getExtensions();
  }

  protected void loadExtension(PullManifestParser parser, 
    Logger logger, ExtensionSource source, int totalCount, int currentCount)
  {
    parser.getContext().getScopeData().put(
      ExtensionVisitor.KEY_EXTENSION_SOURCE, 
      source
    );
    
    URI uri = source.getManifestURI();
    InputStream inStream = null;
    try
    {
      inStream = source.getInputStream();
      parser.parse( inStream, uri.toString() );
    }
    catch ( IOException ioe )
    {
      logger.log( Level.SEVERE, "Error loading manifest from "+ source.getName(), ioe );
    }
    catch ( ParserConfigurationException pce )
    {
      throw new IllegalStateException( "Badly configured jaxb" );
    }
    catch ( XMLStreamException xmlSE )
    {
      logger.log( Level.SEVERE, "Failed to parse manifest from "+source.getName(), 
        xmlSE );
    }
    catch ( RuntimeException re )
    {
      logger.log( Level.SEVERE, "RuntimeException parsing manifest from "+source.getName()+": "+re, re );
    }
    finally
    {
      try { if ( inStream!=null) inStream.close(); } catch ( IOException ioe )
      {
        logger.log( Level.SEVERE, "Error closing stream for "+source.getName(), ioe );
      }
    }
  }
    
  /**
   * Create a logger which will log validation and error messages while
   * parsing manifest files.<p>
   * 
   * This implementation returns a default logger created using the log
   * manager.
   * 
   * @return a logger implementation which will log validation and error
   *    messages.
   */
  protected Logger createExtensionLogger()
  {
    return Logger.getLogger( ExtensionRegistry.class.getName() );
  }

  
  /**
   * Create the element visitor that is responsible for visiting the root
   * element in extension manifests.
   * 
   * @param hookVisitorFactory the visitor factory for the hooks section
   *    of the manifest.
   * @return an element visitor implementation for visiting the root 
   *    element in extension manifests. Typically, this will be a subclass
   *    of ExtensionVisitor.
   */
  protected abstract ExtensionVisitor createExtensionVisitor( 
    ElementVisitorFactory hookVisitorFactory );
  
  /**
   * Add the specified extension source to the classpath, if required.
   * 
   * @param source the source of an extension. E.g if this is a JAR, it 
   *    needs to be added to the classpath.
   * @deprecated since 2.0 this method is no longer called during the
   *    processing of extension sources; the source will be added to the
   *    classpath using the {@link javax.ide.extension.spi.BaseExtensionVisitor#addToClasspath(ElementContext,Extension,URI)} method.
   */
  protected void addToClassPath( ExtensionSource source ) {}
  
  /**
   * Create the visitor factory for hooks. This implementation returns a new
   * instance of javax.ide.extension.spi.DefaultHookVisitorFactory.
   * 
   * @return an ElementVisitorFactory for hooks.
   */
  protected ElementVisitorFactory createHookVisitorFactory()
  {
    return new DefaultHookVisitorFactory();
  }
  
  /**
   * Initializes the extension registry.
   * 
   * This implementation 
   */
  protected void initialize()
  {
    Collection extensions = loadExtensions();
    
    _extensions = new LinkedHashMap<String,Extension>(256);
    for ( Iterator i = extensions.iterator(); i.hasNext(); )
    {
      Extension ext = (Extension) i.next();
      _extensions.put( ext.getID(), ext );
    }
  }

  /**
   * Get the extension registry implementation for this IDE.
   * 
   * @return the extension registry implementation for this ide.
   */
  public static ExtensionRegistry getExtensionRegistry()
  {
    try
    {
      return (ExtensionRegistry) getService( ExtensionRegistry.class );
    }
    catch ( ProviderNotFoundException nse )
    {
      throw new IllegalStateException( "No extension registry", nse );
    }
  }
  
  /**
   * Get the install root of the specified extension. The install root is a 
   * directory containing extension specific files. It is conventionally
   * in the same location as the extension jar and named after the extension 
   * id. 
   * 
   * @param extension an extension to get the install root for. Must not be
   *    null.
   * @return the URI of the install root for the specified extension. This
   *    URI may or may not exist - callers should use 
   *    VirtualFileSystem.getVirtualFileSystem().isDirectory() on the returned
   *    URI to verify that it exists. Must not return null.
   */
  public abstract URI getInstallRoot( Extension extension );
  
  /**
   * Returns the class loader to use for the specified extension. The
   * current context class loader is returned by default.
   * 
   * @param extensionId the id of an extension.
   * @return the class loader for the specified extension. 
   */
  public ClassLoader getClassLoader( String extensionId ) 
  {
    return Thread.currentThread().getContextClassLoader();
  }
}
