/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide;

import java.util.HashMap;
import java.util.Map;
import javax.ide.spi.LookupProvider;
import javax.ide.spi.ProviderNotFoundException;

/**
 * Services provide access to distinct areas of functionality in the IDE for
 * an extension developer.
 */
public abstract class Service 
{
  private boolean _initialized = false;
  private static final Map<Class<?>,Service> _loadedServices = new HashMap<Class<?>,Service>();
  
  /**
   * Initialize this manager. This is called the first time a manager is 
   * retrieved programmatically. The manager should process necessary
   * information from the extension manifest to initialize itself.
   * <p>
   * This implementation does nothing.
   */
  protected void initialize()
  {
    
  }
  
  private static Service findService( Class serviceClass )
    throws ProviderNotFoundException
  {
    Service service = _loadedServices.get( serviceClass );
    if ( service == null )
    {
      service = (Service) LookupProvider.lookup( 
        Thread.currentThread().getContextClassLoader(), serviceClass );
      _loadedServices.put( serviceClass, service );
    }
    return service;
  }
  
  /**
   * Get a service. 
   * 
   * This implementation looks for a META-INF/services/className resource on
   * the classpath of the context classloader. If such a resource exists, it
   * must contain the fully qualified class name of an implementation class
   * of the specified class.
   * 
   * @param serviceClass the class of the service.
   * @return the service.
   * @throws ProviderNotFoundException if an implementation of the service could not
   *    be located.
   */
  protected static Service getService( Class serviceClass )
    throws ProviderNotFoundException
  {
    Service service = findService( serviceClass );
    
    if ( !service._initialized )
    {
      service._initialized = true;
      service.initialize();
    }

    return service;
  }
  
  /**
   * Resets all services, causing them to be recreated from scratch next
   * time the {@link #getService( Class )} API is used. <p>
   * 
   * Extension developers are <b>strongly discouraged</b> from using this
   * method. It is called by the internal LookupProvider API to reset
   * services when a new provider is registered.
   * 
   * @since 2.0
   */
  public static void resetAllServices()
  {
    _loadedServices.clear();
  }
  
  /**
   * Clones loaded services table.
   * 
   * <p>This should only ever be called from test code.</p>
   * 
   * @return the stored table of active servcices
   */
  public static HashMap<Class<?>,Service> cloneLoadedServices() {
    return new HashMap<Class<?>,Service>(_loadedServices);
  }
  
  /**
   * Set the loaded services table.
   * 
   * <p>This should only ever be called from test code.</p>
   * 
   * @param loadedServices the loaded services.
   */
  public static void setLoadedServices(HashMap<Class<?>,Service> loadedServices) {
    _loadedServices.clear();
    _loadedServices.putAll(loadedServices);
  }
}
